/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_open_endpoint.c,v 1.220 2006/12/16 17:11:21 loic Exp $";

#define _GNU_SOURCE /* for cpu affinity */
#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#include "mx__internals.h"
#include "mx__lib.h"
#include "mx__fops.h"
#include "mx__driver_interface.h"
#include "mx__error.h"
#  include "mx__partner.h"
#include "mx__endpoint.h"
#include "mx__mcp_request_ring.h"
#include "mx__stack.h"
#  include "mx__block.h"
#include "mx__memory_pool.h"
#include "mx__request.h"
#include "mx__handle_map.h"
#include "mx_byteswap.h"
#include "mx__regcache.h"
#include "mx_connect.h"
#include "mx__shmem.h"
#include "mx__shim.h"
#include "mx__wait_queue.h"
#include "mx__valgrind.h"
#include "mcp_config.h"
#include "mx__ack.h"
#if MX_OS_LINUX && !defined(MX_KERNEL)
#include <syscall.h>
#include <sched.h>
#endif

static mx_return_t mx__get_board_range(mx_endpt_handle_t handle, uint32_t board_num,
				       uint32_t *start, uint32_t *stop);
static mx_return_t mx__get_endpoint_range(mx_endpt_handle_t handle, uint32_t endpoint_id, 
				   uint8_t *start, uint8_t *stop);
static mx_return_t mx__open_endpoint(uint32_t board_start,
				     uint32_t board_stop,
				     uint32_t *board_num,
				     uint8_t endpoint_start,
				     uint8_t endpoint_stop,
				     uint8_t *endpoint_num,
				     uint32_t *session_id,
				     mx_endpt_handle_t *rethandle);
static mx_return_t mx__process_params(struct mx_endpoint *ep,
				      mx_param_t *params_array,
				      uint32_t params_count);

MX_FUNC(mx_return_t)
mx_open_endpoint(uint32_t board_num,
		 uint32_t endpoint_id,
		 uint32_t endpoint_key,
		 mx_param_t *params_array,
		 uint32_t params_count,
		 mx_endpoint_t *endpoint)
{
  mx_return_t ret = MX_SUCCESS;
  struct mx_endpoint *ep;
  int init_count;
  mx_get_copyblock_t desc;
  uint32_t num_handles;
  int32_t num_rdmas;
  mx_get_nic_id_t x;
  mx_endpt_handle_t handle;
  uint8_t endpoint_start, endpoint_stop, endpoint_num;
  uint32_t board_start, board_stop;
  uint32_t i;
  uint32_t session_id;
  struct mx__partner *partner;
  char board_str[20];
  mx_get_board_val_t board_type;
  uint32_t peer_index;

  MX__MUTEX_LOCK(MX_LOCK);
  init_count = Mx_init_count;

  if (init_count <= 0) {
    ret = MX_NOT_INITIALIZED;
    goto abort_with_nothing;
  }

  /* be sure to start with something zeroed to simplify
     error handling */
  ep = mx_calloc(1, sizeof(*ep));
  if (ep == NULL) {
    ret = MX_NO_RESOURCES;
    goto abort_with_nothing;
  }

  ep->handle = MX__INVALID_HANDLE;
#if !defined MX_KERNEL && !MX_OS_WINNT
  if (mx__opt.no_myrinet) {
    ret = mx__get_endpoint_range(handle, endpoint_id, &endpoint_start, &endpoint_stop);
    if (ret != MX_SUCCESS) {
      goto abort;
    }
    for (i=endpoint_start;i<endpoint_stop;i++) {
      char fname[20];
      int fd;
      sprintf(fname, "/tmp/mxnomyri%d.ep",i);
      fd = open (fname, O_RDWR | O_CREAT, 0644);
      if (lockf (fd, F_TLOCK, 0) == 0)
	goto ep_found;
      close(fd);
    }
    ret = MX_BUSY;
    goto abort;
  ep_found:
    endpoint_num = i;
    ep->eventq = calloc(1, sizeof(mcp_uevt_t));
    ep->eventq_uevt = (mcp_uevt_t*)ep->eventq;
    session_id = 0xdeaddead;
    peer_index = 0;
    ep->wake_pending = 0;
    ep->max_mcp_handles = 100;
    ep->tiny_msg_threshold = MX__MAX_TINY_LEN;
    ep->small_msg_threshold = 1000;
    ep->medium_msg_threshold = 100000;

    if (MX_NEED_KERNEL_WINDOW) {
      static mx_kernel_window_t kwin;
      kwin.hz = 100;
      ep->kernel_window = &kwin;
    }

  } else 
#endif
  {
    mx_lookup_peer_t lookup;
    if ((ret = mx_open_any_board(&handle)) != MX_SUCCESS) {
      goto abort;
    }
    ret = mx__get_board_range(handle, board_num, &board_start, &board_stop);
    if (ret != MX_SUCCESS) {
      mx__close(handle);
      goto abort;
    }
    endpoint_start = endpoint_stop = 0; /* XXX buggy Apple -Wuninitialized */
    ret = mx__get_endpoint_range(handle, endpoint_id, &endpoint_start, &endpoint_stop);
    mx__close(handle);
    if (ret != MX_SUCCESS) {
      goto abort;
    }

    board_num = endpoint_num = session_id = 0; /* XXX buggy Apple -Wuninitialized */
    ret = mx__open_endpoint(board_start, board_stop, &board_num,
			    endpoint_start, endpoint_stop, &endpoint_num,
			    &session_id, &handle);
    if (ret != MX_SUCCESS) {
      goto abort;
    }
    ep->handle = handle; 
    ep->board_num = board_num;
  
    ep->error_handler = 0;
#ifdef MX_UNEXP_QUEUE_LENGTH_MAX
    ep->unexp_queue_length_max = MX_UNEXP_QUEUE_LENGTH_MAX;
#endif
    ret = mx__process_params(ep, params_array, params_count);
    if (ret != MX_SUCCESS) {
      goto abort;
    }
    if (mx__get_num_handles(ep->handle, &num_handles) != 0) {
      ret = mx__error_noep("open_endpoint/get_num_handles", MX_BAD_BAD_BAD);
      goto abort;
    }
    board_type.board_number = board_num;
    ret = mx__get_board_type(ep->handle, &board_type);
    if (ret) {
      ret = mx__error_noep("open_endpoint/get_board_type", MX_BAD_BAD_BAD);
      goto abort;
    }
    ep->board_type = board_type.val;
    ep->is_ze = (ep->board_type == MX_BOARD_TYPE_Z);
    ep->max_mcp_handles = num_handles;
    if (mx__get_max_rdma_windows(ep->handle, &num_rdmas) != 0) {
      ret = mx__error_noep("open_endpoint/get_max_rdma_windows", MX_BAD_BAD_BAD);
      goto abort;
    }
    ret = mx_rdma_init(&ep->rdmas, num_rdmas);
    if (ret != MX_SUCCESS) {
      goto abort;
    }

    ep->rdma_requests = mx_calloc(sizeof(ep->rdma_requests[0]), num_rdmas);
    if (!ep->rdma_requests != MX_SUCCESS) {
      goto abort;
    }
    if (mx__get_copyblocks(ep->handle, &desc) != 0) {
      ret = MX_NO_RESOURCES;
      goto abort;
    }
    mx_memcpy(&ep->desc, &desc, sizeof(desc));

    MX_DEBUG_PRINT(MX_DEBUG_OPEN_CLOSE, ("sendq offset = 0x%x\n", desc.sendq_offset));
    MX_DEBUG_PRINT(MX_DEBUG_OPEN_CLOSE, ("eventq offset = 0x%x\n", desc.eventq_offset));
    MX_DEBUG_PRINT(MX_DEBUG_OPEN_CLOSE, ("sendq len = 0x%x\n", desc.sendq_len));
    MX_DEBUG_PRINT(MX_DEBUG_OPEN_CLOSE, ("eventq len = 0x%x\n", desc.eventq_len));
  
    MX_DEBUG_PRINT(MX_DEBUG_OPEN_CLOSE, ("user_mmapped_sram offset = 0x%x\n", 
				       desc.user_mmapped_sram_offset));
    MX_DEBUG_PRINT(MX_DEBUG_OPEN_CLOSE, ("user_mmapped sram_len = 0x%x\n", 
				       desc.user_mmapped_sram_len));
  
    MX_DEBUG_PRINT(MX_DEBUG_OPEN_CLOSE, ("user_reqq offset = 0x%x\n", desc.user_reqq_offset));
    MX_DEBUG_PRINT(MX_DEBUG_OPEN_CLOSE, ("user_dataq offset = 0x%x\n", desc.user_dataq_offset));
    MX_DEBUG_PRINT(MX_DEBUG_OPEN_CLOSE, ("user_reqq len = 0x%x\n", desc.user_reqq_len));
    MX_DEBUG_PRINT(MX_DEBUG_OPEN_CLOSE, ("user_dataq len = 0x%x\n", desc.user_dataq_len));

    if ((ret = mx__map(ep)) != 0) {
      goto abort;
    }
    
    if (!(ep->kernel_window && ep->kernel_window->hz) && MX_NEED_KERNEL_WINDOW) {
      ret = mx__error_noep("Need kernel window/not provided by driver", MX_FAILURE);
      goto abort;
    }
    ep->eventq_length = desc.eventq_len;
    mx__endpoint_init_eventq(ep);

    ep->recvq_length = desc.recvq_len;
    mx__endpoint_init_recvq(ep);

    if (desc.user_reqq_len / sizeof(mcp_ureq_t) < ep->max_mcp_handles + 1) {
      ret = mx__error_noep("max_mcp_handles + 1 > user_reqq_len", MX_FAILURE);
      goto abort;
    }
    ep->req_ring = mx_malloc(sizeof (*ep->req_ring));
    if (ep->req_ring == NULL) {
      ret = MX_NO_RESOURCES;
      goto abort;
    }
    if (ep->is_ze) {
      mx__mcp_request_ring_init(ep->req_ring,
				ep->ze_req,
				(1 << MX_MCP_VPAGE_SHIFT));
    } else {
      mx__mcp_request_ring_init(ep->req_ring,
				ep->sram+desc.user_reqq_offset,
				desc.user_reqq_len);
    }
    ep->udataq = ep->sram + desc.user_dataq_offset;
    /* TODO: Should the offset come from desc? */
    ep->flow = (uint64_t*) (ep->udataq + ep->desc.user_dataq_len);

    if (mx__get_small_message_threshold(ep->handle, &ep->small_msg_threshold) 
	!= 0) {
      ret = MX_BAD_BAD_BAD;
      goto abort;
    }
    if (mx__get_medium_message_threshold(ep->handle, &ep->medium_msg_threshold)
	!= 0) {
      ret = MX_BAD_BAD_BAD;
      goto abort;
    }
    ep->tiny_msg_threshold = MX__MAX_TINY_LEN;
    mx_always_assert(ep->small_msg_threshold <= MX__MAX_SMALL_LEN);

    if (mx__opt.pipeline_log == -1) {
      if (ep->is_ze) {
	/* 10G pipeline */
	for (i = 0; i < 5; i++)
	  ep->medium_msg_pipelines[i] = 2;
	for (i = 5; i < 64; i++)
	  ep->medium_msg_pipelines[i] = 1;
      } else {
	/* 2G pipeline */
	for (i = 0; i < 8; i++)
	  ep->medium_msg_pipelines[i] = 2;
	for (i = 8; i < 16; i++)
	  ep->medium_msg_pipelines[i] = 1;
	for (i = 16; i < 64; i++)
	  ep->medium_msg_pipelines[i] = 0;
      }
    } else {
      if (mx__opt.pipeline_log > 2 || mx__opt.pipeline_log < 0) {
	ret = MX_BAD_BAD_BAD;
	goto abort;
      }
      for (i = 0; i < 64; i++)
	ep->medium_msg_pipelines[i] = mx__opt.pipeline_log;
    }

    ep->small_msg_ptrs = mx__ptr_stack_alloc(ep->desc.user_dataq_len /
					     ep->small_msg_threshold);
    if (ep->small_msg_ptrs == NULL) {
      ret = MX_NO_RESOURCES;
      goto abort;
    }

    for (i = 0; i < ep->desc.user_dataq_len/ep->small_msg_threshold; ++i) {
      mx__ptr_stack_push(ep->small_msg_ptrs, 
			 ep->udataq + i*ep->small_msg_threshold);
    }
    mx_always_assert(ep->small_msg_threshold <= MX__MAX_SMALL_LEN);

    ep->send_pool = mx_malloc(sizeof (*ep->send_pool));
    if (ep->send_pool == NULL) {
      ret = MX_NO_RESOURCES;
      goto abort;
    }
    mx__memory_pool_init(ep->send_pool, ep->sendq, ep->desc.sendq_len,
			 ep->medium_msg_threshold);
    
    if (mx__get_max_peers(ep->handle, &ep->max_peers) != 0) {
      ret = MX_BAD_BAD_BAD;
      goto abort;
    }
    if (mx__get_max_endpoints(ep->handle, &ep->max_endpoints) != 0) {
      ret = MX_BAD_BAD_BAD;
      goto abort;
    }
    x.board_number = board_num;
    if (mx__get_nic_id(ep->handle, &x) != 0) {
      ret = MX_BAD_BAD_BAD;
      goto abort;
    }
    lookup.nic_id = x.nic_id;
    lookup.board_number = board_num;
    if (mx__nic_id_to_peer_index(ep->handle, &lookup) != 0) {
      mx_fatal("Can't find myself in peer table");
    }
    peer_index = lookup.index;
  }

  ep->ctxid_max = 1ULL << ep->ctxid_bits;
  ep->ctxid_mask = ((uint64_t)ep->ctxid_max - 1) << ep->ctxid_shift;
  if(mx__rl_init(&ep->req_lookaside) != 0) {
    goto abort;
  }

  ep->handle_map = mx__hm_init(ep->max_mcp_handles + 1, 1);
  if (ep->handle_map == NULL) {
    goto abort;
  }

  ep->remote_ep = mx_calloc (ep->max_peers * ep->max_endpoints,
			     sizeof (struct mx__partner*));
  if (ep->remote_ep == NULL) {
    mx_fatal("mx_opend_endpoint:remote_ep allocation failure");
    /* TODO: Error handling. */
  }
  for (i = 0; i < ep->max_peers * ep->max_endpoints; ++i) {
    ep->remote_ep[i] = NULL;
  }

  ep->ctxid = mx_calloc (sizeof(*ep->ctxid), ep->ctxid_max);
  if (ep->ctxid == NULL)
    goto abort;
  for(i = 0; i < ep->ctxid_max; i++) {
    mx__init_request_queue(&ep->ctxid[i].recv_reqq);
    mx__init_request_queue(&ep->ctxid[i].doneq);
    mx__init_request_queue(&ep->ctxid[i].unexpq);
    mx__wait_queue_init(&ep->ctxid[i].peek_queue_head);
    mx__wait_queue_init(&ep->ctxid[i].probe_queue_head);
  }
  mx__wait_queue_init(&ep->wait_queue_head);

  mx__init_request_queue(&ep->send_reqq);
  mx__init_request_queue(&ep->resend_reqq);
  mx__init_request_queue(&ep->buffered_sendq);
  mx__init_request_queue(&ep->large_sendq);
  mx__init_request_queue(&ep->notifying_large_sendq);
  mx__init_request_queue(&ep->large_getq);
  mx__init_request_queue(&ep->multifrag_recvq);
  mx__init_request_queue(&ep->mcp_connectq);
  mx__init_request_queue(&ep->mcp_connect_replyq);
  mx__init_request_queue(&ep->resend_list);
  mx__init_request_queue(&ep->ackq);
  TAILQ_INIT(&ep->partners_to_ack);
  ep->resend_delay = mx_jiffies_hz(ep) / 2 + 1;
  ep->ack_delay = mx_jiffies_hz(ep) / 100 + 1;

  ep->endpoint_sid_n = htonl(session_id);

  /* create and automatically connect a partner to ourself */
  partner = mx__endpoint_lookup_partner(ep, endpoint_num, peer_index);
  mx_fixme_assert(partner);
  partner->endpoint_sid_n = ep->endpoint_sid_n;
  partner->best_session_n = partner->endpoint_sid_n;
  partner->connect_session_n = partner->endpoint_sid_n;
  ep->myself = partner;

#if MX_USE_SHMEM
  if (!mx__opt.disable_shmem) {
    ep->shm = mx_calloc(1,sizeof(*ep->shm));
    ep->shm->peers = mx_calloc(ep->max_endpoints, sizeof(ep->shm->peers[0]));
    ep->shm->shmq = mx__shm_open(ep,endpoint_num, 1, 0);
    mx_fixme_assert(ep->shm->shmq != NULL);
    /* don't start from zero, because bit MX__SHM_REQ_CNT is the one
       used for new type, and type are zero at init */
    MX_VALGRIND_MEMORY_MAKE_NOACCESS(&ep->shm->shmq->queue[0], sizeof(struct mx__shmreq));
    MX_VALGRIND_MEMORY_MAKE_READABLE(&ep->shm->shmq->queue[0].type, sizeof(ep->shm->shmq->queue[0].type));
    ep->shm->shmq->read_idx = MX__SHM_REQ_CNT;
    ep->shm->shmq->write_idx = MX__SHM_REQ_CNT;
    ep->shm->shmq->app_key = endpoint_key;
    mx_fixme_assert(ep->shm->shmq);
    ep->shm->shmq->pid = getpid();
    STAILQ_INIT(&ep->shm->bounce_reqq);
  }
#endif

  mx_rdmawin_init(ep);

#if MX_OS_UDRV
 {
   char pc[MX_MAX_STR_LEN];
   mx_get_info(ep, MX_PRODUCT_CODE, &board_num, sizeof(board_num), pc, MX_MAX_STR_LEN);
   if (strncmp(pc, "SIMULATOR", 9) == 0)
     ep->lxgdb = 1;
   mx_printf("OS_UDRV running in %s mode\n", ep->lxgdb ? "LXGDB" : "HW");
 }
#endif

  ep->timeout = mx__opt.max_retries * ep->resend_delay;

  *endpoint = ep;

  MX__MUTEX_INIT(&ep->lock);
  MX__EVENT_INIT(&ep->in_handler_event);

  MX__MUTEX_LOCK(&Mx_rcache_lock);
  ep->next = Mx_endpoints;
  Mx_endpoints = ep;
  MX__MUTEX_UNLOCK(&Mx_rcache_lock);

  MX__MUTEX_LOCK(&ep->lock);
#if MX_OS_LINUX && !defined(MX_KERNEL) && defined CPU_SETSIZE
  if (mx__opt.cpus) {
    unsigned cpu = endpoint_num % mx__opt.cpus;
    cpu_set_t cpuset;
    CPU_ZERO(&cpuset);
    CPU_SET(cpu, &cpuset);
    syscall(__NR_sched_setaffinity, 0, sizeof(cpuset), (void*)&cpuset);
  }
#endif
  if (!mx__opt.monothread) {
    MX__THREAD_CREATE(&ep->thread, mx__progress_thread, ep);
  }
  mx__init_connect(ep, endpoint_key, params_array, params_count);
  mx__conservation_of_matter(ep);
  MX__MUTEX_UNLOCK(&ep->lock);
  MX__MUTEX_UNLOCK(MX_LOCK);
  return ret;

 abort:
  if (ep->ctxid)
    mx_free(ep->ctxid);
  if (ep->remote_ep)
    mx_free(ep->remote_ep);
  /* mx__rl_fini accept non initialized stuff (if it was zeroed) */
  mx__rl_fini(&ep->req_lookaside);
  if (ep->send_pool)
    mx_free(ep->send_pool);
  if (ep->req_ring)
    mx_free(ep->req_ring);
  if (ep->sendq)
    mx__unmap(ep);
  if (ep->rdmas.max)
    mx_rdma_finalize(&ep->rdmas);
  if (ep->rdma_requests)
    mx_free(ep->rdma_requests);
  if (ep->handle != MX__INVALID_HANDLE)
    mx__close(ep->handle);
  mx_free(ep);
 abort_with_nothing:
  MX__MUTEX_UNLOCK(MX_LOCK);
  sprintf(board_str,"%d", board_num);
  return mx__error_noep("mx_open_endpoint(%s,%d)",ret, 
			board_num == MX_ANY_NIC ? "any" : board_str, endpoint_id);
}

static mx_return_t
mx__get_board_range(mx_endpt_handle_t handle, uint32_t board_num, uint32_t *start, uint32_t *stop)
{
  *start = 0;
  if (mx__get_instance_count(handle, stop) != 0) {
    return  MX_BAD_BAD_BAD;
  }
  if (board_num != MX_ANY_NIC) {
    if (board_num >= *stop) {
      return MX_BOARD_UNKNOWN;
    }
    *start = board_num;
    *stop = board_num + 1;
  }
  return MX_SUCCESS;
}

static mx_return_t
mx__get_endpoint_range(mx_endpt_handle_t handle, uint32_t endpoint_id, uint8_t *start, uint8_t *stop)
{
  mx_return_t ret;
  uint32_t max_ep;
  if (mx__opt.no_myrinet) {
    max_ep = 10;
  } else {
    if ((ret =  mx__get_max_endpoints(handle, &max_ep))) {
      return ret;
    }
  }
  if (endpoint_id == MX_ANY_ENDPOINT) {
    *start = 0;
    *stop = max_ep;
    return MX_SUCCESS;
  } else {
    *start = (uint8_t)endpoint_id;
    *stop = (uint8_t)endpoint_id+1;
    return endpoint_id < max_ep ? MX_SUCCESS : MX_BAD_ENDPOINT;
  }
}

static mx_return_t
mx__open_endpoint(uint32_t board_start, uint32_t board_stop,
		  uint32_t *board_num, uint8_t endpoint_start,
		  uint8_t endpoint_stop, uint8_t* endpoint_num,
		  uint32_t *session_id, mx_endpt_handle_t *rethandle)
{
  mx_endpt_handle_t handle;
  mx_return_t ret, ret_def;
  int driver_ok, dev_exist, perm_ok;
  uint32_t board;
  uint8_t endpoint;

  /* default stuff */
  driver_ok = 0;
  dev_exist = 0;
  perm_ok = 0;
  ret_def = MX_NO_RESOURCES;

  for (endpoint = endpoint_start; endpoint < endpoint_stop; ++endpoint) {
    for (board = board_start; board < board_stop; ++board) {

      /* Try to open this board/endpoing combo */
      ret = mx__open(board, endpoint, &handle);

      if (ret == MX_SUCCESS) {
#ifdef MX_KERNEL
	ret = mx_klib_set_endpoint(&handle, board, endpoint, session_id);
#else
	mx_set_endpt_t set_endpt;
	set_endpt.endpoint = endpoint;
	ret = mx__set_endpoint(handle, &set_endpt);
	*session_id = set_endpt.session_id;
#endif
	if (ret != 0) {
	  ret_def = ret;
	  mx__close(handle);
	  dev_exist = driver_ok = perm_ok = 1;
	}
	else {
	  *board_num = board;
	  *endpoint_num = endpoint;
	  *rethandle = handle;
	  return MX_SUCCESS;
	}
      } else {
        if (ret != MX_NO_DEV) {
	  dev_exist = 1;

	  if (ret != MX_NO_DRIVER) {
	    driver_ok = 1;

	    if (ret != MX_NO_PERM) {
	      perm_ok = 1;
	      ret_def = ret;
	    }
	  }
	}
      }
    }
  }

  if (!dev_exist) {
    ret = MX_NO_DEV;
  } else if (!driver_ok) {
    ret = MX_NO_DRIVER;
  } else if (!perm_ok) {
    ret = MX_NO_PERM;
  } else {
    ret = ret_def;
  }
  return ret;
}

static mx_return_t
mx__process_params(struct mx_endpoint *ep, mx_param_t *params_array,
		   uint32_t params_count)
{
  uint32_t loop;

  if (!params_array && params_count)
    return mx__error(ep, "mx__process_params", MX_BAD_PARAM_LIST);

  for (loop = 0; loop < params_count; ++loop) {
    switch (params_array[loop].key) {
    case MX_PARAM_ERROR_HANDLER:
      ep->error_handler = params_array[loop].val.error_handler;
      break;
    case MX_PARAM_UNEXP_QUEUE_MAX:
      ep->unexp_queue_length_max = params_array[loop].val.unexp_queue_max;
      break;
    case MX_PARAM_CONTEXT_ID:
      if (params_array[loop].val.context_id.bits > MX_CONTEXT_ID_BITS_MAX
	  || params_array[loop].val.context_id.bits + params_array[loop].val.context_id.shift > 64)
	return MX_BAD_PARAM_VAL;
      ep->ctxid_bits = params_array[loop].val.context_id.bits;
      ep->ctxid_shift = params_array[loop].val.context_id.shift;
      break;
    default:
      /* TODO: Complain if we don't understand the param? */
      return mx__error(ep, "mx__process_params", MX_BAD_PARAM_NAME);
    }
  }

  /* unexp_queue_length_max and context_id cannot be activated at the
   * same until we find a scalable way to fix mx__virtual_unexp_length_too_big
   */
  mx_always_assert(!(ep->unexp_queue_length_max && ep->ctxid_bits));

  return MX_SUCCESS;
}


#ifdef MX_VERSIONED_SYMS
/* binary compatibility with previous version */

mx_return_t
mx_open_endpoint_v1(uint64_t nic_id,
		    uint32_t endpoint_id,
		    uint32_t endpoint_key,
		    mx_param_t *params_array,
		    uint32_t params_count,
		    mx_endpoint_t *endpoint)
{
  uint32_t board_num;
  mx_return_t rc;
  mx_printf("mx_open_endpoint called by binary compiled with old mx library\n");
  if ((uint32_t)nic_id != MX_ANY_NIC) {
    rc = mx_nic_id_to_board_number(nic_id, &board_num);
    if (rc != MX_SUCCESS)
      return rc;
    mx_printf("board number = %d\n", board_num);
  } else {
    board_num = MX_ANY_NIC;
  }
  return mx_open_endpoint(board_num, endpoint_id, endpoint_key, 
			  params_array, params_count, endpoint);
}

__asm__(".symver mx_open_endpoint_v1,mx_open_endpoint@MX_0.0");

#endif


